function [hh] = mkart(yt,xt)
%MKART transform map to mercator projection.
%
%      MKART(PHI,LAM), where PHI and LAM are vectors for tickmarks,
%      adds a frame and tickmarks to the current plot and applies a
%      mercator transformation.
%
%      H = MKART(PHI,LAM) also returns the handle vector to the map's
%      frame.
%
%
%      Example:
%               plot(rand(1,10)*180-90,rand(1,10)*360-180,'o');
%               mkart([-80:20:80],[-180:30:180]);
%               grid
%
%
%      NOTE: MKART does not work for images!
%            MKART uses the figures' user data to store scaling information.
%
%      See also  MERC, DEC2DEG, WYSIWYG
%  Uses the following functions: merc, dec2deg, extrem 
%
%  (c) 09-07-1996 Wolfgang Erasmi, Dept. Marine Physics, IfM Kiel, Germany
%  werasmi@ifm.uni-kiel.de
% make shure Latitude is within range
hax = gca;
if nargin < 1,   
   yt=[get(hax,'ytick')];
   if yt(1)> min(get(hax,'ylim'))  yt=[min(get(hax,'ylim')) yt]; end
   if yt(end)< max(get(hax,'ylim'))  yt=[yt max(get(hax,'ylim'))]; end
%   yt=[min(get(hax,'ylim')) get(hax,'ytick')  max(get(hax,'ylim'))];
end
yt=yt(find(yt>-90&yt<90)); 
if length(yt)<1; yt, error('Wrong Latitude values'); end;
if min(diff(yt))<=0; yt, error('Latitude values must be increasing'); end; 

if nargin < 2,   
   xt=[get(hax,'xtick')];
   if xt(1)> min(get(hax,'xlim'))  xt=[min(get(hax,'xlim')) xt]; end
   if xt(end)< max(get(hax,'xlim'))  xt=[xt max(get(hax,'xlim'))]; end
end
if length(xt)<1; xt, error('Wrong Longitude values'); end;
if min(diff(xt))<=0; xt, error('Longitude values must be increasing'); end; 


ref = 0;
ck = round([1 1 1] - get(gcf,'color'));
%transform yt, treat for numerical rounding errors
TT=round(sign(yt).*merc(abs(yt))*1e9)/1e9;
LL=dec2deg(yt,'phi');

orgun=get(gca,'units');

% make map a mercator projection

% check if it already is
haxuserd=get(hax,'userdata');
if length(haxuserd)>7,
   if strcmp(haxuserd(1:8),'mercator'),
      disp('(Already in Mercator projection.)');
      posstr=sscanf(haxuserd(9:length(haxuserd(:))),'%g %g %g %g');
      set(gca,'pos',posstr(:)');
      if nargin<2; return; end
      axis([extrem(xt) merc(extrem(yt))]);
   end;
else
   hch = get(hax,'children');
   % check whether a child is of illegal type 'image':
   for k = length(hch):-1:1,
      if strcmp(get(hch(k),'type'),'image'),
         error('Plot contains object(s) of type image, for which mkart does not work!');
      end;
   end;
   axis([extrem(xt) extrem(TT)]);
   set(gca,'asp',[nan nan]);
   set(gca,'units','norm');
   orgpos=get(gca,'pos');
   % transform all children's ydata:
   for k = length(hch):-1:1,
      if strcmp(get(hch(k),'type'),'text'),
            yd = get(hch(k),'Position');
            set(hch(k),'Position',[yd(1) merc(yd(2)) yd(3)] );
		else
            yd = get(hch(k),'ydata');
            set(hch(k),'ydata',merc(yd));
      end;
   end;
   posstr=sprintf('%8.6f %8.6f %8.6f %8.6f',orgpos);
   % set userdata to indicate  mercator projection and orig. pos. of plot
   set(hax,'userdata',['mercator ',sprintf(posstr,orgpos)]);
end;

set(gca,'units','cent');
len=get(gca,'pos');

rsoll=1;
xlim=get(gca,'xlim');
ylim=get(gca,'ylim');
dx_grd=xlim(2)-xlim(1);
dy_grd=diff(extrem(yt));
dy_val=diff(ylim);
dphi = 1e-3;

aa = diff(merc(ref+[-0.5 0.5]*dphi))/dy_val * len(4);
bb = dphi/dx_grd * len(3);
rist=(aa)/(bb);
fac=rsoll/rist;
pos=get(gca,'pos');
posorg=pos;

% shrink x- or y-axis, depending on factor fac
if fac<=1,
   pos(4)=pos(4) * fac;
   pos(2)=pos(2)+(posorg(4)-pos(4))/2;
else
   pos(3)=pos(3) / fac;
   pos(1)=pos(1)+(posorg(3)-pos(3))/2;
end;
set(gca,'pos',pos)
set(gca,'units',orgun);


% create frame & labels
set(gca,'ytick',TT,'yticklabel',LL,'xtick',xt)
deglabels('x',[-180 180]);

if nargout > 0,
   T=TT; L=LL;
end;

chil=get(gca,'children');
for k=1:length(chil),
   if strcmp(get(chil(k),'userdata'),'mapborder'),
      delete(chil(k));
   end;
end;

set(gca,'tickdir','out','box','on');

xlim=get(gca,'xlim');
ylim=get(gca,'ylim');
ax=axis;

dx=diff(xlim)*0.011;
dy=diff(ylim)*0.011;
if fac<=1,
   dy=dy/fac;
else
   dx=dx*fac;
end;

xx=[xlim(1)-dx xlim(2)+dx xlim(2)+dx xlim(1)-dx xlim(1)-dx];
yy=[ylim(1)-dy ylim(1)-dy ylim(2)+dy ylim(2)+dy ylim(1)-dy];
h1=line(xx,yy,'clip','off','col',ck,'userdata','mapborder');

edgx=[xx(1)+xlim(1) xx(2)+xlim(2)]/2;
edgy=[yy(1)+ylim(1) yy(3)+ylim(2)]/2;

xt=get(gca,'xtick'); i=find((xt>min(xlim)) & (xt<max(xlim)));  xt=[xlim(1) xt(i) xlim(2)];
yt=get(gca,'ytick'); i=find((yt>min(ylim)) & (yt<max(ylim)));  yt=[ylim(1) yt(i) ylim(2)];

XT=[xt(1:length(xt)-1); xt(2:length(xt)); ones(1,length(xt)-1)+nan ];
l=2:2:size(XT,2);
XT(:,l)=[]; xt=reshape(XT,1,prod(size(XT)));
h2=line(xt,ones(size(xt))*edgy(1),'color',ck,'clip','off','userdata','mapborder');
h3=line(xt,ones(size(xt))*edgy(2),'color',ck,'clip','off','userdata','mapborder');

YT=[yt(1:length(yt)-1); yt(2:length(yt)); ones(1,length(yt)-1)+nan ];
l=2:2:size(YT,2);
YT(:,l)=[]; yt=reshape(YT,1,prod(size(YT)));
h4=line(ones(size(yt))*edgx(1),yt,'color',ck,'clip','off','userdata','mapborder');
h5=line(ones(size(yt))*edgx(2),yt,'color',ck,'clip','off','userdata','mapborder');

axis(ax);

if nargout > 0,   hh=[h1;h2;h3;h4;h5]; end;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [yo] = merc(y)
% MERC transfer coordinates according Mercator projection.
%      Yo = MERC(Y) returns the mercator-transformed of Y.

a = 6378137.000;
b = 6356752.314;
e = sqrt(a^2-b^2)/a;

rad = pi/180;

s = e * sin(y*rad);

yo = log(tan(pi./4 + (y*rad)/2) .* (((1-s)./(1+s)) .^ (e/2) ));

yo = real(yo);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [X] = extrem(P);
% EXTREM-werte9
%JH
mi = min(min(P));
ma = max(max(P));
X = [mi ma];


